"use strict";
/**********************************************************************/
/*  Bitsight VRM Archer Integration - Update Archer Vendor Findings   */
/**********************************************************************/

/*
Purpose: 
	To update Archer with Bitsight VRM Findings.

    1. Login to Archer API to obtain session token and Archer version
    2. Obtain necessary Archer backend application, field, and values list id data to construct queries and perform updates due to uniqueness between Archer instances
    3. Create Archer Search XML criteria to identify Third Party Profiles in scope to update with Bitsight VRM data
    4. Get Bitsight list values, get Archer list values, and update Archer list values if necessary
	5. Get all Bitsight VRM vendor findings and data via multiple API calls
	6. Iterate and match between systems and build postbody for Archer if differences found or new
	7. Add or update Bitsight VRM Findings records in Archer

 */

/********************************************************/
/* VERSIONING                                           */
/********************************************************/
/*  
	2/17/2025 - Version 1.0
    Initial Version - 
*/

/********************************************************/
/* LIBRARIES
/********************************************************/
const axios = require("axios");
const fs = require("fs"); //Filesystem
const xmldom = require("@xmldom/xmldom");
const xml2js = require("xml2js");
var { params, ArcherTPPFieldParams, ArcherVendorFindingsFieldParams } = require("./config.js");

/********************************************************/
/* MISC SETTINGS                                        */
/********************************************************/
//Verbose logging will log the post body data and create output files which takes up more disk space
var bVerboseLogging = true;

/********************************************************/
/* GENERAL VARIABLES                                    */
/********************************************************/
//General varibles which should not be changed
var bOverallSuccessful = true; //Optimistic
var sArcherSessionToken = null;

var aArcherTPPReport = []; //Stores the report of records obtained with a Bitsight VRM Domain AND without a Bitsight VRM GUID.
var ArcherValuesLists = []; //This will be used for iterating through to obtain values list value id from Archer
var BitsightVRMFindingStatuses = []; //Stores the Finding Status name and guid after obtaining from Bitsight VRM API
var aArcherBitsightVRMFindings = []; //Stores the main data object of the data from Archer of the Bitsight VRM Findings
var aBitsightVRMFindings = {}; //Stores the main data object of the data from Bitsight of the Bitsight VRM Findings. Key/value pair with array of findings inside.

var sErrorLogDetails = "";
var totalErrors = 0;
var totalWarnings = 0;

var totalArcherTPPs = 0;
var totalArcherBitsightFindings = 0;
var totalBitsightFindings = 0;
var totalSuccessUpdate = 0;
var totalErrorUpdate = 0;

//Used to store files in logs subdirectory
var appPrefix = "120";

//Bitsight Tracking Stats
var COST_ArcherBitsightVersion = "RSA Archer 1.0";
var COST_Platform = "RSA Archer";
var BitsightCustomerName = "unknown";

//Just a simple way to remember the Archer input data types
var ArcherInputTypes = {
	"text": 1,
	"numeric": 2,
	"date": 3,
	"valueslist": 4,
	"externallinks": 7,
	"usergrouprecperm": 8,
	"crossref": 9,
	"attachment": 11,
	"image": 12,
	"matrix": 16,
	"ipaddress": 19,
	"relatedrecords": 23,
	"subform": 24,
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//BEGIN HELPER FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
function LogInfo(text) {
	//Log the details
	console.log(getDateTime() + "::INFO:: " + text);
}

function LogWarn(text) {
	//Log the details
	console.log(getDateTime() + "::WARN:: " + text);

	sErrorLogDetails += getDateTime() + "::WARN:: " + text + "\r\n";
	totalWarnings++;
}

function LogError(text) {
	//Update stat
	totalErrors++;

	//Log overall error to file
	LogSaaSError();

	//Log the details to output log file
	console.log(getDateTime() + "::ERROR:: " + text);

	//Log the error that gets sent to the API monitoring
	sErrorLogDetails += getDateTime() + "::ERROR:: " + text + "\r\n";
}

//Only log this if verbose logging is turned on - not needed unless troubleshooting
function LogVerbose(text) {
	if (bVerboseLogging) {
		console.log(getDateTime() + "::VERB:: " + text);
	}
}

//Simple function to get date and time in standard format
function getDateTime() {
	var dt = new Date();
	return (
		pad(dt.getFullYear(), 4) +
		"-" +
		pad(dt.getMonth() + 1, 2) +
		"-" +
		pad(dt.getDate(), 2) +
		" " +
		pad(dt.getHours(), 2) +
		":" +
		pad(dt.getMinutes(), 2) +
		":" +
		pad(dt.getSeconds(), 2)
	);
}

//Pads a certain amount of characters based on the size of text provided
function pad(num, size) {
	var s = num + "";
	//prepend a "0" until desired size reached
	while (s.length < size) {
		s = "0" + s;
	}
	return s;
}

//Simple method to log an error to a file for batch execution. The batch file will check if this file exists.
function LogSaaSError() {
	if (bOverallSuccessful == true) {
		bOverallSuccessful = false; //Set the flag to false so we only create this file one time and avoid file lock issues.

		fs.writeFileSync("logs\\error-" + appPrefix + ".txt", "ERROR");
		LogInfo("Logged error and created logs\\error-" + appPrefix + ".txt file.");
	}
}

//Simple method to log successful execution for execution. The batch file will check if this file exists.
function LogSaaSSuccess() {
	if (bOverallSuccessful == true) {
		fs.writeFileSync("logs\\success-" + appPrefix + ".txt", "SUCCESS");
		LogInfo("Logged success and created logs\\success-" + appPrefix + ".txt file.");
	}
}

function xmlStringToXmlDoc(xml) {
	var p = new xmldom.DOMParser();
	return p.parseFromString(xml, "text/xml");
	//return p.parseFromString(xml, "application/xml");
}

function b64Encode(str) {
	var CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
	var out = "",
		i = 0,
		len = str.length,
		c1,
		c2,
		c3;

	while (i < len) {
		c1 = str.charCodeAt(i++) & 0xff;
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt((c1 & 0x3) << 4);
			out += "==";
			break;
		}

		c2 = str.charCodeAt(i++);
		if (i == len) {
			out += CHARS.charAt(c1 >> 2);
			out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
			out += CHARS.charAt((c2 & 0xf) << 2);
			out += "=";
			break;
		}

		c3 = str.charCodeAt(i++);
		out += CHARS.charAt(c1 >> 2);
		out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xf0) >> 4));
		out += CHARS.charAt(((c2 & 0xf) << 2) | ((c3 & 0xc0) >> 6));
		out += CHARS.charAt(c3 & 0x3f);
	}
	return out;
}

function makeBasicAuth(token) {
	//Purpose of this is to convert the token to the authorization header for basic auth
	//Format is Token with a colon at the end then converted to Base64
	return b64Encode(token + ":");
}

//GetArcherText validates then returns data from a text value
function GetArcherText(sText) {
	try {
		if (typeof sText == "undefined" || typeof sText._ == "undefined" || sText == null) {
			return "";
		} else {
			return sText._.trim();
		}
	} catch (ex) {
		LogWarn("GetArcherText() Error getting Archer text. ex:" + ex);
		LogVerbose("GetArcherText() Error getting Archer text. ex.stack:" + ex.stack);
		return "";
	}
}

//GetArcherValue validates then returns data from a single value list
function GetArcherValue(jValueNode) {
	try {
		if (
			typeof jValueNode == "undefined" ||
			jValueNode == null ||
			typeof jValueNode.ListValues == "undefined" ||
			typeof jValueNode.ListValues[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue[0]._ == "undefined"
		) {
			return "";
		} else {
			return jValueNode.ListValues[0].ListValue[0]._;
		}
	} catch (ex) {
		LogWarn("GetArcherValue() Error getting Archer single value. ex:" + ex);
		LogVerbose("GetArcherValue() Error getting Archer single value. ex.stack:" + ex.stack);
		return "";
	}
}

//GetArcherValue validates then returns data from a multi-select value list
function GetArcherValues(jValueNode) {
	LogVerbose("GetArcherValues() Start. jValueNode=" + JSON.stringify(jValueNode));

	let tmp = [];
	try {
		if (
			typeof jValueNode == "undefined" ||
			jValueNode == null ||
			typeof jValueNode.ListValues == "undefined" ||
			typeof jValueNode.ListValues[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue.length == 0 ||
			typeof jValueNode.ListValues[0].ListValue[0] == "undefined" ||
			typeof jValueNode.ListValues[0].ListValue[0]._ == "undefined"
		) {
			return [];
		} else {
			//Iterate through the nodes and return an array of values.

			for (let v in jValueNode.ListValues[0].ListValue) {
				tmp[tmp.length] = jValueNode.ListValues[0].ListValue[v]._;
				LogVerbose("GetArcherValues() v=" + v + " Value=" + jValueNode.ListValues[0].ListValue[v]._);
			}
		}
	} catch (ex) {
		LogWarn("GetArcherValues() Error iterating Archer multi-select values list. ex:" + ex);
		LogVerbose("GetArcherValues() Error iterating Archer multi-select values list. ex.stack:" + ex.stack);
	} finally {
		//Doing this in a "finally" block in case there was an error iterating and we had some data.
		LogInfo("GetArcherValues() tmp=" + JSON.stringify(tmp));
		return tmp;
	}
}

//GetArcherReference validates then returns data from a cross-reference field (type=9)
function GetArcherReference(refData) {
	LogVerbose("GetArcherReference() Start. refData=" + JSON.stringify(refData));

	try {
		if (
			typeof refData == "undefined" ||
			refData == null ||
			typeof refData.Reference == "undefined" ||
			typeof refData.Reference[0] == "undefined" ||
			typeof refData.Reference[0]._ == "undefined" ||
			typeof refData.Reference[0].$ == "undefined" ||
			typeof refData.Reference[0].$.id == "undefined"
		) {
			LogInfo("GetArcherReference() refData passed in was empty.");
			return {};
		} else {
			//Get and return the data from the referenece display and the Archer ID of that record as an object
			return {
				"name": refData.Reference[0]._,
				"id": refData.Reference[0].$.id,
			};
		}
	} catch (ex) {
		LogWarn("GetArcherReference() Error reading reference field or id. ex:" + ex);
		LogVerbose("GetArcherReference() Error reference field or id. ex.stack:" + ex.stack);
		return {};
	}
}

function getDateYYYYMMMDD() {
	//Returns current date in YYYY-MM-DD format which is what Archer is expecting
	try {
		var dt = new Date();
		return pad(dt.getFullYear(), 4) + "-" + pad(dt.getMonth() + 1, 2) + "-" + pad(dt.getDate(), 2);
	} catch (ex) {
		LogWarn("getDateYYYYMMMDD() - Issue creating YYYYMMDD date string. Returning emptystring. ex:" + ex.stack);
		return "";
	}
}

function formatDateToYYYYMMDD(dateString) {
	try {
		const date = new Date(dateString);
		const year = date.getFullYear();
		const month = String(date.getMonth() + 1).padStart(2, "0"); // Month is 0-indexed
		const day = String(date.getDate()).padStart(2, "0");
		return `${year}-${month}-${day}`;
	} catch (ex) {
		LogWarn("formatDateToYYYYMMDD() - Issue creating YYYYMMDD date string. Returning emptystring. ex:" + ex.stack);
		return "";
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//END HELPER FUNCTIONS
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////BEGIN CORE FUNCTIONALITY
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

async function ArcherLogin() {
	LogInfo("ArcherLogin() Start.");

	//Optimistic
	let bSuccess = true;
	let data = null;

	try {
		//construct body
		const body = {
			"InstanceName": params["archer_instanceName"],
			"Username": params["archer_username"],
			"UserDomain": "",
			"Password": params["archer_password"],
		};

		const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_loginpath"];

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: {
				"Content-Type": "application/json",
				Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
			},
			data: body,
		};

		LogVerbose("ArcherLogin() API call httpConfig=" + JSON.stringify(httpConfig));

		//API call to get session token
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("ArcherLogin() Axios call complete. data=" + JSON.stringify(data));
				//Expecting data to be in the format of: {"Links":[],"RequestedObject":{"SessionToken":"823B73BA88E36B8DABB113E56DDE9FB8","InstanceName":"Archer_Bitsight67","UserId":212,"ContextType":0,"UserConfig":{"TimeZoneId":"Eastern Standard Time","TimeZoneIdSource":1,"LocaleId":"en-US","LocaleIdSource":2,"LanguageId":1,"DefaultHomeDashboardId":-1,"DefaultHomeWorkspaceId":-1,"LanguageIdSource":1,"PlatformLanguageId":1,"PlatformLanguagePath":"en-US","PlatformLanguageIdSource":1},"Translate":false,"IsAuthenticatedViaRestApi":true},"IsSuccessful":true,"ValidationMessages":[]}
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("ArcherLogin() Axios call error. Err: " + error);
				LogVerbose("ArcherLogin() Axios call error. Err.stack: " + error.stack);
			});

		if (bSuccess === true) {
			//Attempt to get the session token
			if (
				typeof data != "undefined" &&
				data != null &&
				typeof data.RequestedObject != "undefined" &&
				data.RequestedObject != null &&
				typeof data.RequestedObject.SessionToken != "undefined" &&
				data.RequestedObject.SessionToken != null &&
				data.RequestedObject.SessionToken.length > 10
			) {
				sArcherSessionToken = data.RequestedObject.SessionToken;
				LogVerbose("ArcherLogin() SessionToken=" + sArcherSessionToken);
			} else {
				LogError("ArcherLogin() Archer Session Token missing.");
				bSuccess = false;
			}
		}
	} catch (ex) {
		bSuccess = false;
		LogError("ArcherLogin() Error constructing call. ex: " + ex);
		LogVerbose("ArcherLogin() Error constructing call. ex.stack: " + ex.stack);
	}

	//Return bSuccess if it passed or failed. We need the Archer session token to do anything.
	return bSuccess;
}

/********************************************************/
/********	GetArcherVersion
/********************************************************/
async function GetArcherVersion() {
	LogInfo("GetArcherVersion() Start.");
	try {
		let data = null;

		var sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_version"];

		const httpConfig = {
			method: "GET",
			url: sUrl,
			headers: {
				"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
				"Content-Type": "application/json",
				Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
			},
			data: {},
		};

		LogVerbose("GetArcherVersion() API call httpConfig=" + JSON.stringify(httpConfig));

		//API call to get Archer Version
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("GetArcherVersion() Axios call complete. data=" + JSON.stringify(data));
				//Expecting data to be in the format of:
			})
			.catch(function (error) {
				LogWarn("GetArcherVersion() Axios call error. Err: " + error);
				LogVerbose("GetArcherVersion() Axios call error. Err.stack: " + error.stack);
			});

		//If the version is available, update the COST_Platform variable (it is generic by default...this adds the version)
		if (typeof data != "undefined" && typeof data.RequestedObject != "undefined" && data.RequestedObject.Version != "undefined") {
			let sArcherVersionNum = data.RequestedObject.Version; //Get the Version
			COST_Platform = COST_Platform + " (" + sArcherVersionNum + ")";
			LogInfo("sArcherVersionNum: " + sArcherVersionNum);
		} else {
			sArcherVersionNum = "Unknown";
			LogWarn("Unable to obtain Archer Version Number.");
		}
	} catch (ex) {
		//We won't fail on this because it's not urgent and we have default values available.
		LogWarn("GetArcherVersion() Error constructing call or parsing result. ex: " + ex);
		LogVerbose("GetArcherVersion() Error constructing call or parsing result. ex.stack: " + ex.stack);
	}
}

/********************************************************/
/********	getTPPFieldIDs
/********************************************************/
async function getArcherTPPFieldIDs() {
	LogInfo("getArcherTPPFieldIDs() Start.");

	async function getTPPModuleID() {
		LogInfo("getTPPModuleID() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_applicationpath"];
			const odataquery = "?$select=Name,Id,Guid&$filter=Name eq '" + params["archer_ThirdPartyProfileApp"] + "'";
			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getTPPModuleID() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("getTPPModuleID() Axios call complete. data=" + JSON.stringify(data));
					//Expecting data to be in the format of:
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getTPPModuleID() Axios call error. Err: " + error);
					LogVerbose("getTPPModuleID() Axios call error. Err.stack: " + error.stack);
				});

			//If the module is available, use it and get the fields.
			if (typeof data != "undefined" && typeof data[0].RequestedObject != "undefined" && data[0].RequestedObject.Id != "undefined") {
				var iModuleID = data[0].RequestedObject.Id; //Get the content ID
				LogInfo("iModuleID: " + iModuleID);
				//Set as a param variable used later for search queries and record updates.
				params["archer_ThirdPartyProfileAppID"] = iModuleID;

				//Get Field IDs for the app
				bSuccess = await getFieldIDs(iModuleID); //the function will return true or false for success
			} else {
				LogError("ERROR Obtaining Third Party Profile module ID.");
				bSuccess = false;
			}
		} catch (ex) {
			LogError("getTPPModuleID() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getTPPModuleID() Error constructing call or parsing result. ex.stack: " + ex.stack);
			bSuccess = false;
		}

		return bSuccess;
	}

	async function getFieldIDs(iModuleID) {
		LogInfo("getFieldIDs() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_fielddefinitionapppath"] + iModuleID;
			//const odataquery = "?$orderby=Name&$filter=Name eq 'Bitsight Portfolio Created?' or Name eq 'Domain'"; //example
			//Ideally we could filter using an operator like "starts with" or "contains", but Archer doesn't support anything like that.
			//We could have used filters for each field name, but there are size restrictions for the odata query we would exceed.
			//So we will retrieve all fields, then parse to get the ones we care about.
			//Unfortunately Archer won't let us specify the attributes we actually need, so all are returned.
			//const odataquery = "?$select=Name,Id,Guid,LevelId,RelatedValuesListId&$orderby=Name";
			const odataquery = "?$orderby=Name";
			//Obtaining the field name, id, guid, levelId, and the relatedvalueslistId if it's a list.

			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getFieldIDs() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			//NOTE: Fields for ALL levels are returned, so be aware of that if we ever add fields at different levels.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					//LogVerbose("getFieldIDs() Axios call complete. data=" + JSON.stringify(data));

					if (bVerboseLogging === true) {
						let filename = "logs\\" + appPrefix + "\\" + "01getFieldIDs.json";
						fs.writeFileSync(filename, JSON.stringify(data));
						LogVerbose("Saved getFieldIDs data to " + filename);
					}
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getFieldIDs() Axios call error. Err: " + error);
					LogVerbose("getFieldIDs() Axios call error. Err.stack: " + error.stack);
				});

			if (typeof data != "undefined" && typeof data[0].RequestedObject != "undefined") {
				LogVerbose("getFieldIDs() Archer returned good TPP app data.");

				for (let iField in data) {
					let sFieldName = data[iField].RequestedObject.Name.toLowerCase().trim();

					//Uncomment to see all fields evaluated.
					//LogVerbose("*Looking for: " + sFieldName);

					//sField is the field name (key of the json object)
					for (let sField in ArcherTPPFieldParams) {
						//Get the value into lowercase
						let sFieldLower = sField.toLowerCase();

						//Compare the Archer value to the value we have in our ArcherTPPFieldParams object
						if (sFieldName == sFieldLower) {
							//If we have a match, then we'll get the details and set the data to the ArcherTPPFieldParams object

							let sId = data[iField].RequestedObject.Id;
							let sGuid = data[iField].RequestedObject.Guid;
							let sRelatedValuesListId = data[iField].RequestedObject.RelatedValuesListId;

							let tmp = {
								"id": sId,
								"guid": sGuid,
								"RelatedValuesListId": sRelatedValuesListId,
							};

							//Obtain Values Lists
							//If the RelatedValuesListId exists, then add it to the ArcherValuesLists array
							//We have 2 values lists that we'll need to get the values for later (Bitsight VRM Tags and Bitsight VRM Lifecycle).
							if (typeof sRelatedValuesListId != "undefined" && sRelatedValuesListId != null) {
								ArcherValuesLists[ArcherValuesLists.length] = {
									"FieldName": sField,
									"ValuesListID": sRelatedValuesListId,
									"Values": [],
								};
							}

							//Set the ArcherTPPFieldParams value for this field
							ArcherTPPFieldParams[sField] = tmp;

							//get out of this loop
							break;
						}
					}
				}

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "02ArcherTPPFieldParams.json";
					fs.writeFileSync(filename, JSON.stringify(ArcherTPPFieldParams));
					LogVerbose("Saved ArcherTPPFieldParams to " + filename);
				}

				//Normally we would go get the Values List Values, but need the Vendor Findings app valuest list field IDs first and then we can get all of the values lists at the same time.
				return bSuccess;
			} else {
				bSuccess = false;
				LogError("ERROR Obtaining TPP field definitions. Cannot continue.");
			}
		} catch (ex) {
			bSuccess = false;
			LogError("getFieldIDs() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getFieldIDs() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}

		return bSuccess;
	}

	//Start of all the functions to get TPP ids
	return await getTPPModuleID();
}

/********************************************************/
/********	getVendorFindingsFieldIDs
/********************************************************/
async function getVendorFindingsFieldIDs() {
	LogInfo("getVendorFindingsFieldIDs() Start.");

	async function getVendorFindingsModuleID() {
		LogInfo("getVendorFindingsModuleID() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_questionnairepath"];
			const odataquery = "?$select=Name,Id,Guid&$filter=Name eq '" + params["archer_BitsightVRMFindings"] + "'";
			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getVendorFindingsModuleID() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("getVendorFindingsModuleID() Axios call complete. data=" + JSON.stringify(data));
					//Expecting data to be in the format of:
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getVendorFindingsModuleID() Axios call error. Err: " + error);
					LogVerbose("getVendorFindingsModuleID() Axios call error. Err.stack: " + error.stack);
				});

			//If the module is available, use it and get the fields.
			if (typeof data != "undefined" && data != null && data != [] && typeof data[0].RequestedObject != "undefined" && data[0].RequestedObject.Id != "undefined") {
				var iModuleID = data[0].RequestedObject.Id; //Get the content ID
				LogInfo("iModuleID: " + iModuleID);
				//Set as a param variable used later for search queries and record updates.
				params["archer_VendorFindingsAppID"] = iModuleID;

				//Get Level ID for the questionnaire
				bSuccess = await getVendorFindingsLevelID(iModuleID); //the function will return true or false for success
			} else {
				LogError("ERROR Obtaining Bitsight VRM Findings module ID.");
				bSuccess = false;
			}
		} catch (ex) {
			LogError("getVendorFindingsModuleID() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getVendorFindingsModuleID() Error constructing call or parsing result. ex.stack: " + ex.stack);
			bSuccess = false;
		}

		return bSuccess;
	}

	async function getVendorFindingsLevelID(iModuleID) {
		LogInfo("getVendorFindingsLevelID() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_getlevelidpath"] + iModuleID;

			const postBody = {};

			const httpConfig = {
				method: "POST",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					"X-HTTP-Method-Override": "GET",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getVendorFindingsLevelID() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("getVendorFindingsLevelID() Axios call complete. data=" + JSON.stringify(data));
					//Expecting data to be in the format of:
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getVendorFindingsLevelID() Axios call error. Err: " + error);
					LogVerbose("getVendorFindingsLevelID() Axios call error. Err.stack: " + error.stack);
				});

			//If the module is available, use it and get the fields.
			if (typeof data != "undefined" && data != null && data != [] && typeof data[0].RequestedObject != "undefined" && data[0].RequestedObject.Id != "undefined") {
				let iLevelID = data[0].RequestedObject.Id; //Get the content ID
				LogInfo("iLevelID: " + iLevelID);
				//Set as a param variable used later for search queries and record updates.
				params["archer_VendorFindingsLevelID"] = iLevelID;

				//Get Field IDs for the app
				bSuccess = await getVendorFindingsFieldIDs(iModuleID); //the function will return true or false for success
			} else {
				LogError("getVendorFindingsLevelID() ERROR Obtaining Bitsight VRM Findings Level ID.");
				bSuccess = false;
			}
		} catch (ex) {
			LogError("getVendorFindingsLevelID() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getVendorFindingsLevelID() Error constructing call or parsing result. ex.stack: " + ex.stack);
			bSuccess = false;
		}

		return bSuccess;
	}

	async function getVendorFindingsFieldIDs(iModuleID) {
		LogInfo("getVendorFindingsFieldIDs() Start.");
		let bSuccess = true;

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_fielddefinitionapppath"] + iModuleID;
			//const odataquery = "?$orderby=Name&$filter=Name eq 'Bitsight Portfolio Created?' or Name eq 'Domain'"; //example
			//Ideally we could filter using an operator like "starts with" or "contains", but Archer doesn't support anything like that.
			//We could have used filters for each field name, but there are size restrictions for the odata query we would exceed.
			//So we will retrieve all fields, then parse to get the ones we care about.
			//Unfortunately Archer won't let us specify the attributes we actually need, so all are returned.
			//const odataquery = "?$select=Name,Id,Guid,LevelId,RelatedValuesListId&$orderby=Name";
			const odataquery = "?$orderby=Name";
			//Obtaining the field name, id, guid, levelId, and the relatedvalueslistId if it's a list.

			const postBody = {
				"Value": odataquery,
			};

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("getVendorFindingsFieldIDs() API call httpConfig=" + JSON.stringify(httpConfig));

			//API call to get Archer Third Party Profile application module ID.
			//NOTE: Fields for ALL levels are returned, so be aware of that if we ever add fields at different levels.
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					//LogVerbose("getVendorFindingsFieldIDs() Axios call complete. data=" + JSON.stringify(data));

					if (bVerboseLogging === true) {
						let filename = "logs\\" + appPrefix + "\\" + "03getFieldIDs.json";
						fs.writeFileSync(filename, JSON.stringify(data));
						LogVerbose("Saved getFieldIDs data to " + filename);
					}
				})
				.catch(function (error) {
					bSuccess = false;
					LogError("getVendorFindingsFieldIDs() Axios call error. Err: " + error);
					LogVerbose("getVendorFindingsFieldIDs() Axios call error. Err.stack: " + error.stack);
				});

			if (typeof data != "undefined" && typeof data[0].RequestedObject != "undefined") {
				LogVerbose("getVendorFindingsFieldIDs() Archer returned good Bitsight VRM Findings questionnaire data.");

				for (let iField in data) {
					let sFieldName = data[iField].RequestedObject.Name.toLowerCase().trim();

					//Uncomment to see all fields evaluated.
					//LogVerbose("*Looking for: " + sFieldName);

					//sField is the field name (key of the json object)
					for (let sField in ArcherVendorFindingsFieldParams) {
						//Get the value into lowercase
						let sFieldLower = sField.toLowerCase();

						//Compare the Archer value to the value we have in our ArcherVendorFindingsFieldParams object
						if (sFieldName == sFieldLower) {
							//If we have a match, then we'll get the details and set the data to the ArcherVendorFindingsFieldParams object

							let sId = data[iField].RequestedObject.Id;
							let sGuid = data[iField].RequestedObject.Guid;
							let sRelatedValuesListId = data[iField].RequestedObject.RelatedValuesListId;

							let tmp = {
								"id": sId,
								"guid": sGuid,
								"RelatedValuesListId": sRelatedValuesListId,
							};

							//Obtain Values Lists
							//If the RelatedValuesListId exists, then add it to the ArcherValuesLists array
							//We have 2 values lists that we'll need to get the values for later (Bitsight VRM Tags and Bitsight VRM Lifecycle).
							if (typeof sRelatedValuesListId != "undefined" && sRelatedValuesListId != null) {
								ArcherValuesLists[ArcherValuesLists.length] = {
									"FieldName": sField,
									"ValuesListID": sRelatedValuesListId,
									"Values": [],
								};
							}

							//Set the ArcherVendorFindingsFieldParams value for this field
							ArcherVendorFindingsFieldParams[sField] = tmp;

							//get out of this loop
							break;
						}
					}
				}

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "04ArcherVendorFindingsFieldParams.json";
					fs.writeFileSync(filename, JSON.stringify(ArcherVendorFindingsFieldParams));
					LogVerbose("Saved ArcherVendorFindingsFieldParams to " + filename);
				}

				//Get the values list values
				bSuccess = await getValuesListValues(0);
			} else {
				bSuccess = false;
				LogError("getVendorFindingsFieldIDs() ERROR Obtaining Vendor Findings field definitions. Cannot continue.");
			}
		} catch (ex) {
			bSuccess = false;
			LogError("getVendorFindingsFieldIDs() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getVendorFindingsFieldIDs() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}

		return bSuccess;
	}

	//Gets values list values for both TPP and Vendor Findings because it doesn't matter what app you request it from.
	async function getValuesListValues(currentValuesList) {
		let bSuccess = true;
		LogInfo("----------------------------getValuesListValues----------------------------");
		let data;
		try {
			LogInfo("getValuesListValues() Getting VL: " + ArcherValuesLists[currentValuesList].FieldName);
			const valueslistID = ArcherValuesLists[currentValuesList].ValuesListID;
			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_valueslistvaluepath"] + valueslistID;

			const httpConfig = {
				method: "GET",
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: {},
			};

			LogInfo("getValuesListValues() httpConfig: " + JSON.stringify(httpConfig));

			//API call to get values list
			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("getValuesListValues() Axios call complete. data=" + JSON.stringify(data));
				})
				.catch(function (error) {
					//Catch 500 error. Archer produces an HTTP 500 error if there are no values in a values list instead of simply returning an empty object.
					if (error.response) {
						if (error.response.status === 500) {
							LogInfo("getValuesListValues() Values list is empty (Archer sends an error, but this was caught). List name: " + ArcherValuesLists[currentValuesList].FieldName);
						} else {
							bSuccess = false;
							LogError("getValuesListValues() Axios call error. Err: " + error);
							LogVerbose("getValuesListValues() Axios call error. Err.stack: " + error.stack);
						}
					}
				});

			//Parse results
			if (bSuccess === true && typeof data != "undefined" && typeof data[0].RequestedObject != "undefined") {
				LogVerbose("getValuesListValues() Archer returned good values list data.");

				let id = "";
				let name = "";

				for (let i in data) {
					id = data[i].RequestedObject.Id;
					name = data[i].RequestedObject.Name;

					// //Format name of value to remove spaces and /
					// name = name.replace(/\//g, "");
					// name = name.replace(/ /g, "");

					let tmp = {
						"name": name,
						"id": id,
					};

					//Set the list info to the next item in the ArcherValuesLists.Values item
					ArcherValuesLists[currentValuesList].Values[ArcherValuesLists[currentValuesList].Values.length] = tmp;
				}
			} else {
				LogVerbose("getValuesListValues() Archer returned empty data.");
			}

			//Iterate through if there were multiple
			//Did we hit the max?
			LogVerbose("getValuesListValues() Current=" + currentValuesList + " Max=" + ArcherValuesLists.length);
			if (currentValuesList >= ArcherValuesLists.length - 1) {
				LogVerbose("getValuesListValues() Hit maxResults of " + ArcherValuesLists.length);
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "ArcherValuesLists.json", JSON.stringify(ArcherValuesLists));
					LogVerbose("Saved to logs\\" + appPrefix + "\\" + "ArcherValuesLists.json file");
				}
				return bSuccess;
			} else {
				//Still have more values lists to iterate through...
				currentValuesList++; //Increment before running again
				LogVerbose("getValuesListValues() Iterating through next ValuesList=" + currentValuesList);
				bSuccess = await getValuesListValues(currentValuesList);
			}
		} catch (ex) {
			bSuccess = false;
			LogError("getValuesListValues() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("getValuesListValues() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}

		return bSuccess;
	}

	//Start of all the functions to get TPP ids
	return await getVendorFindingsModuleID();
}

async function getBitsightListValues() {
	LogInfo("getBitsightListValues() Start.");
	let bSuccess = true;

	bSuccess = await getBitsightListValues_FindingStatus();
	//May need more in the future
	return bSuccess;
}

async function getBitsightListValues_FindingStatus() {
	LogInfo("getBitsightListValues_FindingStatus() Start.");
	let bSuccess = true;

	try {
		const sURL = params["Bitsight_webroot"] + params["Bitsight_getfindingstatusesapi"]; //Should be: https://service.Bitsighttech.com/customer-api/vrm/v1/findings/statuses

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		const options = {
			method: "GET",
			url: sURL,
			headers: headers,
			data: {},
		};

		LogVerbose("getBitsightListValues_FindingStatus() API options=" + JSON.stringify(options));

		try {
			const { data } = await axios.request(options);
			//LogVerbose("getBitsightListValues_FindingStatus() data=" + JSON.stringify(data));

			if (typeof data == "undefined" || data == null || typeof data.results == "undefined" || data.results == null) {
				LogError("getBitsightListValues_FindingStatus() API returned undefined or null.");
				bSuccess = false;
			} else {
				//Iterate the statuses and populate the object

				for (let i in data.results) {
					//LogVerbose("getBitsightListValues_FindingStatus() i=" + i + " " + data[i].name + "=" + data[i].guid);

					BitsightVRMFindingStatuses[BitsightVRMFindingStatuses.length] = {
						"name": data.results[i].name,
						"guid": data.results[i].guid,
					};
				}

				//Note that it is possible that there are no statuses, but unlikely. Our test instance had 2 default statuses.

				//Save to file...
				if (bVerboseLogging === true) {
					var fs = require("fs");
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "BitsightVRMFindingStatuses.json", JSON.stringify(BitsightVRMFindingStatuses));
					LogInfo("getBitsightListValues_FindingStatus() Saved to logs\\" + appPrefix + "\\" + "BitsightVRMFindingStatuses.json file");
				}
			}
		} catch (ex) {
			LogError("getBitsightListValues_FindingStatus() Axios call error. ex: " + ex);
			LogVerbose("getBitsightListValues_FindingStatus() Axios call error. ex.stack: " + ex.stack);
			bSuccess = false;
		}
	} catch (ex) {
		LogError("getBitsightListValues_FindingStatus() Error constructing api call. ex: " + ex);
		LogVerbose("getBitsightListValues_FindingStatus() Error constructing api call. ex.stack: " + ex.stack);
		bSuccess = false;
	}
	return bSuccess;
}

async function ValidateAndUpdateListValues() {
	LogInfo("ValidateAndUpdateListValues() Start - Finding Status.");
	let bSuccess = true;
	try {
		//Check Bitsight Findings - iterate and if new values are found, add them to Archer
		for (let b in BitsightVRMFindingStatuses) {
			let sBitsightFindingStatusValue = BitsightVRMFindingStatuses[b].name;
			LogVerbose("ValidateAndUpdateListValues() Looking for sBitsightFindingStatusValue=" + sBitsightFindingStatusValue);
			let sArcherValuesListID = null;

			//Archer Value List...Find the correct list
			for (let a in ArcherValuesLists) {
				if (ArcherValuesLists[a].FieldName == "Finding Status") {
					sArcherValuesListID = ArcherValuesLists[a].ValuesListID;
					LogVerbose("ValidateAndUpdateListValues()   Found 'Bitsight Finding Status' in ArcherValuesLists");
					let bFound = false;
					//Archer Values in the list...
					for (let aValue in ArcherValuesLists[a].Values) {
						//Does the Bitsight value match the Archer list?
						let sArcherFindingStatusValue = ArcherValuesLists[a].Values[aValue].name;
						//LogVerbose("ValidateAndUpdateListValues()      Current value: sArcherFindingStatusValue=" + sArcherFindingStatusValue);
						if (sArcherFindingStatusValue == sBitsightFindingStatusValue) {
							//we need the Archer Values List ID to add the new value
							LogVerbose("ValidateAndUpdateListValues()      **MATCH FOUND** " + sBitsightFindingStatusValue);
							bFound = true;
							break;
						}
					}
					try {
						if (bFound === false) {
							LogVerbose("ValidateAndUpdateListValues()      **NOT FOUND** Add:" + sBitsightFindingStatusValue);
							let newValuesListID = await CreateArcherListValue(sArcherValuesListID, sBitsightFindingStatusValue);
							if (newValuesListID != null && newValuesListID > 0) {
								//Need to add to the Archer values list values
								ArcherValuesLists[a].Values[ArcherValuesLists[a].Values.length] = {
									"name": sBitsightFindingStatusValue,
									"id": newValuesListID,
								};
							}
						}
					} catch (ex) {
						LogWarn("ValidateAndUpdateListValues() Unable to update ArcherValuesLists with new values list value. ex: " + ex);
						LogVerbose("ValidateAndUpdateListValues() Unable to update ArcherValuesLists with new values list value. ex.stack: " + ex.stack);
					}
				}
			}
		}
	} catch (ex) {
		LogError("ValidateAndUpdateListValues() Error iterating loops. ex: " + ex);
		LogVerbose("ValidateAndUpdateListValues() Error iterating loops. ex.stack: " + ex.stack);
		bSuccess = false;
	}

	//Output the updated values list file for troubleshooting.
	if (bVerboseLogging === true) {
		var fs = require("fs");
		fs.writeFileSync("logs\\" + appPrefix + "\\" + "ArcherValuesListsUpdated.json", JSON.stringify(ArcherValuesLists));
		LogVerbose("Saved ArcherValuesLists to logs\\" + appPrefix + "\\" + "ArcherValuesListsUpdated.json file");
	}

	return bSuccess;
}

async function CreateArcherListValue(sListNumber, sNewValue) {
	LogInfo("CreateArcherListValue() Start. sListNumber=" + sListNumber + " sNewValue=" + sNewValue);
	let bSuccess = true;
	try {
		let sXML =
			'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchemainstance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
			'<soap:Body><CreateValuesListValue xmlns="http://archer-tech.com/webservices/"><sessionToken>' +
			sArcherSessionToken +
			"</sessionToken><valuesListId>" +
			sListNumber +
			"</valuesListId><valuesListValueName>" +
			sNewValue +
			"</valuesListValueName></CreateValuesListValue></soap:Body></soap:Envelope>";

		//Must escape the XML to next inside of the soap request...
		//sXML = sXML.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;");

		LogVerbose("CreateArcherListValue() sXML=" + sXML);

		/* build url */
		var sUrl = params["archer_webroot"] + params["archer_ws_root"] + params["archer_fieldpath"];

		var headers = {
			"content-type": "text/xml; charset=utf-8",
			"SOAPAction": "http://archer-tech.com/webservices/CreateValuesListValue",
		};

		// const postBody =
		// 	'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
		// 	"<soap:Body>" +
		// 	sXML +
		// 	"</CreateValuesListValue></soap:Body></soap:Envelope>";

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: headers,
			data: sXML,
		};

		//Execute API Query
		LogVerbose("CreateArcherListValue() API call httpConfig=" + JSON.stringify(httpConfig));
		let data;
		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("CreateArcherListValue() Axios call complete. data=" + data);
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("CreateArcherListValue() Axios call error. Err: " + error);
				LogVerbose("CreateArcherListValue() Axios call error. Err.stack: " + error.stack);
			});

		if (bSuccess === true) {
			try {
				//Convert XML data results to an XMLDOM for parsing
				let doc = xmlStringToXmlDoc(data);

				if (bSuccess === true && typeof doc != "undefined" && doc != null) {
					const resultElement = doc.getElementsByTagName("CreateValuesListValueResult")[0];

					if (resultElement) {
						const sArcherValuesListValueID = resultElement.textContent;
						LogInfo("CreateArcherListValue() Result Values List Value ID:" + sArcherValuesListValueID); // Output: Result Value: 81301
						//Success! Return the ID
						const iArcherValuesListValueID = parseInt(sArcherValuesListValueID, 10); // Radix 10 is important!
						return iArcherValuesListValueID;
					} else {
						LogError("CreateArcherListValue() element not found.");
						return null;
					}
				}
			} catch (ex) {
				LogError("CreateArcherListValue() Error parsing new Archer values list value. ex: " + ex);
				LogVerbose("CreateArcherListValue() Error parsing new Archer values list value. ex.stack: " + ex.stack);
				return null;
			}
		} else {
			return null;
		}
	} catch (ex) {
		LogError("CreateArcherListValue() Error constructing api call. ex: " + ex);
		LogVerbose("CreateArcherListValue() Error constructing api call. ex.stack: " + ex.stack);
		return null;
	}
	//If we got here, we didn't receive a new values list ID.
	return null;
}

//Function to build search criteria and get report of TPPs so we can update the findings
async function getArcherTPPsWithDomain() {
	LogInfo("getArcherTPPsWithDomain() Start.");
	let bSuccess = true;
	let data = null;
	try {
		//Construct search query based on data obtained an populated in the ArcherTPPFieldParams object (it has the guids for the fields)
		let sSearchCriteria =
			"<SearchReport><PageSize>10000</PageSize><MaxRecordCount>10000</MaxRecordCount>" +
			'<DisplayFields><DisplayField name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			'</DisplayField><DisplayField name="Bitsight GUID">' +
			ArcherTPPFieldParams["Bitsight GUID"].guid +
			'</DisplayField><DisplayField name="Bitsight VRM GUID">' +
			ArcherTPPFieldParams["Bitsight VRM GUID"].guid +
			'</DisplayField></DisplayFields><Criteria><ModuleCriteria><Module name="Third Party Profile">' +
			params["archer_ThirdPartyProfileAppID"] +
			'</Module><SortFields><SortField name="Sort1"><Field name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			"</Field><SortType>Ascending</SortType></SortField></SortFields></ModuleCriteria><Filter>" +
			'<Conditions><TextFilterCondition name="Text 1"><Field name="Bitsight VRM Domain">' +
			ArcherTPPFieldParams["Bitsight VRM Domain"].guid +
			"</Field><Operator>DoesNotEqual</Operator><Value></Value></TextFilterCondition></Conditions>" +
			"</Filter></Criteria></SearchReport>";

		LogVerbose("getArcherTPPsWithDomain() sSearchCriteria=" + sSearchCriteria);

		/* build url */
		var sUrl = params["archer_webroot"] + params["archer_ws_root"] + params["archer_searchpath"];

		var headers = {
			"content-type": "text/xml; charset=utf-8",
			"SOAPAction": "http://archer-tech.com/webservices/ExecuteSearch",
		};

		//Must escape the XML to next inside of the soap request...
		sSearchCriteria = sSearchCriteria.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;");

		const postBody =
			'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
			'<soap:Body><ExecuteSearch xmlns="http://archer-tech.com/webservices/"><sessionToken>' +
			sArcherSessionToken +
			"</sessionToken>" +
			"<searchOptions>" +
			sSearchCriteria +
			"</searchOptions><pageNumber>1</pageNumber></ExecuteSearch></soap:Body></soap:Envelope>";

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: headers,
			data: postBody,
		};

		//Execute API Search Query
		LogVerbose("getArcherTPPsWithDomain() API call httpConfig=" + JSON.stringify(httpConfig));

		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("getArcherTPPsWithDomain() Axios call complete. data=" + data);

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "05getArcherTPPsWithDomain.xml";
					fs.writeFileSync(filename, data);
					LogVerbose("Saved getArcherTPPsWithDomain data to " + filename);
				}
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("getArcherTPPsWithDomain() Axios call error. Err: " + error);
				LogVerbose("getArcherTPPsWithDomain() Axios call error. Err.stack: " + error.stack);
			});

		//Need to parse the data here, but want to take a look at it first.
	} catch (ex) {
		bSuccess = false;
		LogError("getArcherTPPsWithDomain() Error constructing call or parsing result. ex: " + ex);
		LogVerbose("getArcherTPPsWithDomain() Error constructing call or parsing result. ex.stack: " + ex.stack);
	}

	if (bSuccess === true) {
		bSuccess = await parseArcherTPPRecords(data);
	}

	return bSuccess;
}

async function parseArcherTPPRecords(data) {
	LogInfo("parseArcherTPPRecords() Start.");
	let bSuccess = true;
	//variable for our json object
	let resultJSON = null;

	try {
		//Convert XML data results to an XMLDOM for parsing
		let doc = xmlStringToXmlDoc(data);

		//Check to see if nothing was returned from the search query
		if (
			typeof doc.getElementsByTagName("ExecuteSearchResult") == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue == "undefined"
		) {
			//There weren't any records
			LogInfo("parseArcherTPPRecords() No TPPs with a Bitsight VRM Domain AND without a Bitsight VRM GUID. Exiting 01.");
			//aArcherTPPReport will remain empty as a result, but technically this was successful.
			return bSuccess;
		} //Need to proceed and check the count anyway.
		else {
			//let tmp = new xmldom.XMLSerializer().serializeToString(doc);
			//console.log("----------------------------------------------------------------------------------------------------------------------");
			//console.log("doc=" + tmp);

			//Need to get the xml inside the SOAP request and url decode the results
			//ORIG:var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue;
			let sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].textContent; //New method after upgrading xmldom

			//console.log("----------------------------------------------------------------------------------------------------------------------");
			//console.log("sXML=" + sXML);

			// tmp = new xmldom.XMLSerializer().serializeToString(sXML);
			// console.log("----------------------------------------------------------------------------------------------------------------------");
			// console.log("sXMLParsed=" + tmp);

			//turn the xml results into the json object
			xml2js.parseString(sXML, function (err, result) {
				resultJSON = result; //get the result into the object we can use below;
			});
			//console.log("----------------------------------------------------------------------------------------------------------------------");

			//let JSONText = JSON.stringify(resultJSON);
			console.log("resultJSON=" + JSON.stringify(resultJSON));
			//console.log("resultJSON=" + JSONText);

			if (bVerboseLogging === true) {
				let filename = "logs\\" + appPrefix + "\\" + "06archerTPPs.json";
				let fs = require("fs");
				fs.writeFileSync(filename, JSON.stringify(resultJSON));
				LogVerbose("Saved resultJSON data to " + filename);
			}

			let iNumCompanies = resultJSON.Records.$.count; //Get the number of record returned
			LogInfo("iNumCompanies=" + iNumCompanies);

			//Set overall stats
			totalArcherTPPs = parseInt(iNumCompanies);

			//Check to see if we have any existing records
			if (iNumCompanies == 0) {
				LogInfo("parseArcherTPPRecords() No TPPs with a Bitsight VRM Domain AND without a Bitsight VRM GUID. Exiting 02.");
				//aArcherTPPReport will remain empty as a result, but technically this was successful.
				//This will happen the majority of the time for this application.
				return bSuccess;
			}
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing Archer result. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing Archer result. ex.stack: " + ex.stack);
		return bSuccess;
	}

	try {
		params["archer_ThirdPartyProfileLevelID"] = resultJSON.Records.LevelCounts[0].LevelCount[0].$.id;
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing TPP Archer level id data. Cannot continue. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing TPP Archer level id data. Cannot continue. ex.stack: " + ex.stack);
		return bSuccess;
	}

	//If we got this far, we didn't have errors AND we have data.
	try {
		let iID_TPPBitsightVRMDomain;
		let iID_TPPBitsightVRMGUID;
		let iID_TPPBitsightGUID;

		let sName;
		let sID;
		//let sAlias;
		//Iterate through the FieldDefinition to get the field ids that we care about
		for (let h in resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition) {
			sName = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.name;
			//sAlias = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.alias;
			sID = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.id;

			//LogInfo("  Alias:" + sAlias + "   ID: " + sID);
			if (sName == "Bitsight VRM Domain") {
				iID_TPPBitsightVRMDomain = sID;
			} else if (sName == "Bitsight VRM GUID") {
				iID_TPPBitsightVRMGUID = sID;
			} else if (sName == "Bitsight GUID") {
				iID_TPPBitsightGUID = sID;
			}
		}

		let sTPPContentID = "";
		let sTPPBitsightVRMDomain;
		let sTPPBitsightVRMGUID;
		let sTPPBitsightGUID;

		//Iterate through each Archer TPP record....
		for (let i in resultJSON.Records.Record) {
			LogInfo("-----ARCHER TPP RECORD #" + i + "-------");
			sTPPContentID = resultJSON.Records.Record[i].$.contentId;
			sTPPBitsightVRMDomain = "";
			sTPPBitsightVRMGUID = "";
			sTPPBitsightGUID = "";

			//Iterate through the Field elements for the current config record to get the goodies
			for (let y in resultJSON.Records.Record[i].Field) {
				//Get the id of the field because we need to match on the ones we care about...
				sID = resultJSON.Records.Record[i].Field[y].$.id;

				//Now find all the good data we care about...
				if (sID == iID_TPPBitsightVRMDomain) {
					sTPPBitsightVRMDomain = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightVRMGUID) {
					sTPPBitsightVRMGUID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_TPPBitsightGUID) {
					sTPPBitsightGUID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
			}

			LogInfo("Content ID: " + sTPPContentID + " sTPPBitsightVRMDomain: " + sTPPBitsightVRMDomain);
			//Populate the main record with the details we care about....
			aArcherTPPReport[aArcherTPPReport.length] = {
				"ArcherContentID": sTPPContentID,
				"BitsightVRMDomain": sTPPBitsightVRMDomain,
				"sTPPBitsightVRMGUID": sTPPBitsightVRMGUID,
				"sTPPBitsightGUID": sTPPBitsightGUID,
				"APIStatus": "Ready",
			};

			LogInfo("*TPP Number#" + aArcherTPPReport.length + "=" + JSON.stringify(aArcherTPPReport[aArcherTPPReport.length - 1]));
		}

		//Save to file...
		if (bVerboseLogging === true) {
			var fs = require("fs");
			fs.writeFileSync("logs\\" + appPrefix + "\\" + "aArcherTPPReport.json", JSON.stringify(aArcherTPPReport));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "aArcherTPPReport.json file");
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherTPPRecords() Error parsing Archer data. ex: " + ex);
		LogVerbose("parseArcherTPPRecords() Error parsing Archer data. ex.stack: " + ex.stack);
	}

	return bSuccess;
}

//Function to build search criteria and get report of Archer VRM Findings so we can compare and update the findings
async function getArcherVRMFindings() {
	LogInfo("getArcherVRMFindings() Start.");
	let bSuccess = true;
	let data = null;
	try {
		//Construct search query based on data obtained an populated in the ArcherTPPFieldParams object (it has the guids for the fields)
		let sSearchCriteria =
			"<SearchReport><PageSize>10000</PageSize><MaxRecordCount>10000</MaxRecordCount>" +
			'<DisplayFields><DisplayField name="Related Third Party Profile">' +
			ArcherVendorFindingsFieldParams["Related Third Party Profile"].guid +
			'</DisplayField><DisplayField name="Finding Status">' +
			ArcherVendorFindingsFieldParams["Finding Status"].guid +
			'</DisplayField><DisplayField name="Criticality">' +
			ArcherVendorFindingsFieldParams["Criticality"].guid +
			'</DisplayField><DisplayField name="Will Remediate By">' +
			ArcherVendorFindingsFieldParams["Will Remediate By"].guid +
			'</DisplayField><DisplayField name="Bitsight Finding ID">' +
			ArcherVendorFindingsFieldParams["Bitsight Finding ID"].guid +
			'</DisplayField><DisplayField name="Creation Date">' +
			ArcherVendorFindingsFieldParams["Creation Date"].guid +
			'</DisplayField><DisplayField name="Reported On">' +
			ArcherVendorFindingsFieldParams["Reported On"].guid +
			'</DisplayField></DisplayFields><Criteria><ModuleCriteria><Module name="Bitsight VRM Findings">' +
			params["archer_VendorFindingsAppID"] +
			"</Module><SortFields /></ModuleCriteria><Filter><Conditions /></Filter></Criteria></SearchReport>";

		LogVerbose("getArcherVRMFindings() sSearchCriteria=" + sSearchCriteria);

		/* build url */
		var sUrl = params["archer_webroot"] + params["archer_ws_root"] + params["archer_searchpath"];

		var headers = {
			"content-type": "text/xml; charset=utf-8",
			"SOAPAction": "http://archer-tech.com/webservices/ExecuteSearch",
		};

		//Must escape the XML to next inside of the soap request...
		sSearchCriteria = sSearchCriteria.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&apos;");

		const postBody =
			'<?xml version="1.0" encoding="utf-8"?><soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">' +
			'<soap:Body><ExecuteSearch xmlns="http://archer-tech.com/webservices/"><sessionToken>' +
			sArcherSessionToken +
			"</sessionToken>" +
			"<searchOptions>" +
			sSearchCriteria +
			"</searchOptions><pageNumber>1</pageNumber></ExecuteSearch></soap:Body></soap:Envelope>";

		const httpConfig = {
			method: "POST",
			url: sUrl,
			headers: headers,
			data: postBody,
		};

		//Execute API Search Query
		LogVerbose("getArcherVRMFindings() API call httpConfig=" + JSON.stringify(httpConfig));

		await axios(httpConfig)
			.then(function (res) {
				data = res.data;
				LogVerbose("getArcherVRMFindings() Axios call complete. data=" + data);

				if (bVerboseLogging === true) {
					let filename = "logs\\" + appPrefix + "\\" + "07getArcherVRMFindings.xml";
					fs.writeFileSync(filename, data);
					LogVerbose("Saved getArcherVRMFindings data to " + filename);
				}
			})
			.catch(function (error) {
				bSuccess = false;
				LogError("getArcherVRMFindings() Axios call error. Err: " + error);
				LogVerbose("getArcherVRMFindings() Axios call error. Err.stack: " + error.stack);
			});

		//Need to parse the data here, but want to take a look at it first.
	} catch (ex) {
		bSuccess = false;
		LogError("getArcherVRMFindings() Error constructing call or parsing result. ex: " + ex);
		LogVerbose("getArcherVRMFindings() Error constructing call or parsing result. ex.stack: " + ex.stack);
	}

	if (bSuccess === true) {
		bSuccess = await parseArcherVRMFindingsRecords(data);
	}

	return bSuccess;
}

async function parseArcherVRMFindingsRecords(data) {
	LogInfo("parseArcherVRMFindingsRecords() Start.");
	let bSuccess = true;
	//variable for our json object
	let resultJSON = null;

	try {
		//Convert XML data results to an XMLDOM for parsing
		let doc = xmlStringToXmlDoc(data);

		//Check to see if nothing was returned from the search query
		if (
			typeof doc.getElementsByTagName("ExecuteSearchResult") == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue == "undefined"
		) {
			//There weren't any records
			LogInfo("parseArcherVRMFindingsRecords() No Bitsight VRM Findings. Exiting 01.");
			//aArcherBitsightVRMFindings will remain empty as a result, but technically this was successful. And it is expected for a new environment.
			return bSuccess;
		} //Need to proceed and check the count anyway.
		else {
			//let tmp = new xmldom.XMLSerializer().serializeToString(doc);
			//console.log("----------------------------------------------------------------------------------------------------------------------");
			//console.log("doc=" + tmp);

			//Need to get the xml inside the SOAP request and url decode the results
			//ORIG:var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue;
			let sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].textContent; //New method after upgrading xmldom

			//console.log("----------------------------------------------------------------------------------------------------------------------");
			//console.log("sXML=" + sXML);

			// tmp = new xmldom.XMLSerializer().serializeToString(sXML);
			// console.log("----------------------------------------------------------------------------------------------------------------------");
			// console.log("sXMLParsed=" + tmp);

			//turn the xml results into the json object
			xml2js.parseString(sXML, function (err, result) {
				resultJSON = result; //get the result into the object we can use below;
			});
			//console.log("----------------------------------------------------------------------------------------------------------------------");

			//let JSONText = JSON.stringify(resultJSON);
			console.log("resultJSON=" + JSON.stringify(resultJSON));
			//console.log("resultJSON=" + JSONText);

			if (bVerboseLogging === true) {
				let filename = "logs\\" + appPrefix + "\\" + "08aArcherBitsightVRMFindingsRaw.json";
				let fs = require("fs");
				fs.writeFileSync(filename, JSON.stringify(resultJSON));
				LogVerbose("Saved resultJSON data to " + filename);
			}

			let iNumArcherBitsightVRMFindings = resultJSON.Records.$.count; //Get the number of record returned
			LogInfo("iNumArcherBitsightVRMFindings=" + iNumArcherBitsightVRMFindings);

			//Set overall stats
			totalArcherBitsightFindings = parseInt(iNumArcherBitsightVRMFindings);

			//Check to see if we have any existing records
			if (iNumArcherBitsightVRMFindings == 0) {
				LogInfo("parseArcherVRMFindingsRecords() No Bitsight VRM Findings. Exiting 02.");
				//aArcherBitsightVRMFindings will remain empty as a result, but technically this was successful. And it is expected for a new environment.
				return bSuccess;
			}
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherVRMFindingsRecords() Error parsing Archer result. ex: " + ex);
		LogVerbose("parseArcherVRMFindingsRecords() Error parsing Archer result. ex.stack: " + ex.stack);
		return bSuccess;
	}

	//Attempt to get the level id because we'll need it for any adds/updates.
	try {
		params["archer_VendorFindingsLevelID"] = resultJSON.Records.LevelCounts[0].LevelCount[0].$.id;
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherVRMFindingsRecords() Error parsing Archer Bitsight VRM Findings level id data. Cannot continue. ex: " + ex);
		LogVerbose("parseArcherVRMFindingsRecords() Error parsing Archer Bitsight VRM Findings level id data. Cannot continue. ex.stack: " + ex.stack);
		return bSuccess;
	}

	//If we got this far, we didn't have errors AND we have data.
	try {
		let iID_BitsightFinding_Related3rdPartyProfile;
		let iID_BitsightFinding_FindingStatus;
		let iID_BitsightFinding_Criticality;
		let iID_BitsightFinding_WillRemediateBy;
		let iID_BitsightFinding_BitsightFindingID;
		let iID_BitsightFinding_CreationDate;
		let iID_BitsightFinding_ReportedOn;

		let sName;
		let sID;
		//let sAlias;
		//Iterate through the FieldDefinition to get the field ids that we care about
		for (let h in resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition) {
			sName = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.name;
			//sAlias = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.alias;
			sID = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.id;

			//LogInfo("  Alias:" + sAlias + "   ID: " + sID);
			if (sName == "Related Third Party Profile") {
				iID_BitsightFinding_Related3rdPartyProfile = sID;
			} else if (sName == "Finding Status") {
				iID_BitsightFinding_FindingStatus = sID;
			} else if (sName == "Criticality") {
				iID_BitsightFinding_Criticality = sID;
			} else if (sName == "Will Remediate By") {
				iID_BitsightFinding_WillRemediateBy = sID;
			} else if (sName == "Bitsight Finding ID") {
				iID_BitsightFinding_BitsightFindingID = sID;
			} else if (sName == "Creation Date") {
				iID_BitsightFinding_CreationDate = sID;
			} else if (sName == "Reported On") {
				iID_BitsightFinding_ReportedOn = sID;
			}
		}

		let sContentID = "";
		let sBitsightFinding_Related3rdPartyProfile;
		let sBitsightFinding_FindingStatus;
		let sBitsightFinding_Criticality;
		let sBitsightFinding_WillRemediateBy;
		let sBitsightFinding_BitsightFindingID;
		let sBitsightFinding_CreationDate;
		let sBitsightFinding_ReportedOn;

		//Iterate through each Archer Finding record....
		for (let i in resultJSON.Records.Record) {
			LogInfo("-----ARCHER Bitsight VRM Finding RECORD #" + i + "-------");
			sContentID = resultJSON.Records.Record[i].$.contentId;
			sBitsightFinding_Related3rdPartyProfile = "";
			sBitsightFinding_FindingStatus = "";
			sBitsightFinding_Criticality = "";
			sBitsightFinding_WillRemediateBy = "";
			sBitsightFinding_BitsightFindingID = "";
			sBitsightFinding_CreationDate = "";
			sBitsightFinding_ReportedOn = "";

			//Iterate through the Field elements for the current config record to get the goodies
			for (let y in resultJSON.Records.Record[i].Field) {
				//Get the id of the field because we need to match on the ones we care about...
				sID = resultJSON.Records.Record[i].Field[y].$.id;

				//Now find all the good data we care about...
				if (sID == iID_BitsightFinding_Related3rdPartyProfile) {
					sBitsightFinding_Related3rdPartyProfile = GetArcherReference(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_BitsightFinding_FindingStatus) {
					sBitsightFinding_FindingStatus = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_BitsightFinding_Criticality) {
					sBitsightFinding_Criticality = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_BitsightFinding_WillRemediateBy) {
					sBitsightFinding_WillRemediateBy = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_BitsightFinding_BitsightFindingID) {
					sBitsightFinding_BitsightFindingID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_BitsightFinding_CreationDate) {
					sBitsightFinding_CreationDate = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				} else if (sID == iID_BitsightFinding_ReportedOn) {
					sBitsightFinding_ReportedOn = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
			}

			let sName = "**UNKNOWN** - No reference?";
			if (typeof sBitsightFinding_Related3rdPartyProfile != "undefined" && typeof sBitsightFinding_Related3rdPartyProfile.name != "undefined") {
				sName = sBitsightFinding_Related3rdPartyProfile.name;
			}
			LogInfo("Content ID: " + sContentID + " sName: " + sName);
			//Populate the main record with the details we care about....
			aArcherBitsightVRMFindings[aArcherBitsightVRMFindings.length] = {
				"ArcherContentID": sContentID,
				"sBitsightFinding_Related3rdPartyProfile": sBitsightFinding_Related3rdPartyProfile, //This is an {} object with name and id attributes
				"sBitsightFinding_FindingStatus": sBitsightFinding_FindingStatus,
				"sBitsightFinding_Criticality": sBitsightFinding_Criticality,
				"sBitsightFinding_WillRemediateBy": sBitsightFinding_WillRemediateBy,
				"sBitsightFinding_BitsightFindingID": sBitsightFinding_BitsightFindingID,
				"sBitsightFinding_CreationDate": sBitsightFinding_CreationDate,
				"sBitsightFinding_ReportedOn": sBitsightFinding_ReportedOn,
				"APIStatus": "Ready",
			};

			//This could produce a lot of text with many findings so should consider commenting out. Especially since we output it to a file when done iterating.
			LogVerbose("*ArcherBitsightFinding Number#" + aArcherBitsightVRMFindings.length + "=" + JSON.stringify(aArcherBitsightVRMFindings[aArcherBitsightVRMFindings.length - 1]));
		}

		//Save to file...
		if (bVerboseLogging === true) {
			var fs = require("fs");
			fs.writeFileSync("logs\\" + appPrefix + "\\" + "aArcherBitsightVRMFindings.json", JSON.stringify(aArcherBitsightVRMFindings));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "aArcherBitsightVRMFindings.json file");
		}
	} catch (ex) {
		bSuccess = false;
		LogError("parseArcherVRMFindingsRecords() Error parsing Archer data. ex: " + ex);
		LogVerbose("parseArcherVRMFindingsRecords() Error parsing Archer data. ex.stack: " + ex.stack);
	}

	return bSuccess;
}

async function GetBitsightVRMFindings() {
	LogInfo("GetBitsightVRMFindings() Start.");

	try {
		const sURL = params["Bitsight_webroot"] + params["Bitsight_vendorfindingsapi"]; //Should be: https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/findings/query
		//Documentation: https://Bitsight.stoplight.io/docs/VRM-api/uxk1yjtssm95w-list-vendors-findings

		const sAuth = "Basic " + makeBasicAuth(params["Bitsight_token"]);

		const headers = {
			"Content-Type": "application/json",
			"Accept": "application/json",
			"Authorization": sAuth,
			"X-Bitsight-CONNECTOR-NAME-VERSION": COST_ArcherBitsightVersion,
			"X-Bitsight-CALLING-PLATFORM-VERSION": COST_Platform,
			"X-Bitsight-CUSTOMER": BitsightCustomerName,
		};

		//Expected response for 1 record limit and 4 records total to iterate: {"offset":0,"limit":10,"count":4,"results":[{"guid":"0add4431-bdc5-4459-a50c-3c3d6bd83115","review_status":"Awaiting Response","review_status_guid":"7e06eacf-4a49-4a9b-ac10-f33c39ead1c3","review_status_type":2,"vendor_guid":"96f2d60f-e86a-45ad-b5b0-ce33e1538377","created_at":"2025-02-06T16:17:47.279932","criticality":"LOW","description":"Test Finding From Bitsight Support","artifact_guid":"f1de9118-165d-4002-9e81-393bc0446ca7","will_be_resolved":null,"remediation_date":null,"last_modified":"2025-02-17 13:52:19","reported_on":"CAIQ v4 - Audit & Assurance","short_guid":"83115","linking_guids":{"survey_id":"99054c0a-a49b-4e61-8487-43cb84a3ea35","category_id":"f4f93ba8-ee22-4683-9c4a-47d4dedddf1d"},"messages_summary":null,"will_remediate_by":"pending","artifact_type":"questionnaire","is_past_due":false,"vendor":null,"enterprise_guid":"bc750dfc-1670-4ba2-9589-a813722d927d"}]}
		//offset = current page
		//limit = number of records that we want to obtain at a time. However, the JSON response always resets this to 10 no matter how many were requested.
		//count = number of records available...Not pages, so we need to do math to figure out the number of calls.

		//Example result when done iterating: {"offset":0,"limit":10,"count":4,"results":[]}
		//So if the results attribute is an empty array, we can quit.

		const iBitsightvrmapi_limit = params["Bitsightvrmapi_limit"];

		try {
			//Outer loop of iterating the Archer TPP Vendors
			for (let iArcherCompany in aArcherTPPReport) {
				const sTPPBitsightVRMGUID = aArcherTPPReport[iArcherCompany].sTPPBitsightVRMGUID;
				const sBitsightVRMDomain = aArcherTPPReport[iArcherCompany].BitsightVRMDomain;

				let iTotalCompanies = aArcherTPPReport.length;

				LogInfo("GetBitsightVRMFindings()****************************************************************************");
				LogInfo("Getting data for #" + iArcherCompany + " of " + iTotalCompanies - 1 + " domain=" + sBitsightVRMDomain + " GUID=" + sTPPBitsightVRMGUID);
				LogInfo("GetBitsightVRMFindings()****************************************************************************"); //Current page of Bitsight Findings

				//Now attempt to pull the vendor findings until the results is []
				//reset offset
				let iCurrentOffset = 0;

				//Create shell for key/value pair with the findings per domain
				aBitsightVRMFindings[sBitsightVRMDomain] = {
					"findings": [],
				};

				let bMoreData = true;
				while (bMoreData === true) {
					try {
						const postBody = {
							"sort": ["-created_at"],
							"offset": iCurrentOffset,
							"limit": iBitsightvrmapi_limit,
							"vendor_guid": sTPPBitsightVRMGUID,
						};

						const options = {
							method: "POST",
							url: sURL,
							headers: headers,
							data: postBody,
						};

						LogVerbose("GetBitsightVRMFindings() API postBody=" + JSON.stringify(postBody));

						let { data } = await axios.request(options);

						//Check for lack of data or no more results
						if (typeof data == "undefined" || typeof data.results == "undefined" || data.results == null || data.results == [] || data.results.length == 0) {
							LogInfo("GetBitsightVRMFindings() No data returned, continuing.");
							bMoreData = false;
						} else {
							//Got data!

							//Parse and get the results into object.
							//LogVerbose("GetBitsightVRMFindings() Data returned:" + JSON.stringify(data, 4));

							totalBitsightFindings = totalBitsightFindings + (await ParseBitsightVRMVendorFindings(sBitsightVRMDomain, sTPPBitsightVRMGUID, data));

							iCurrentOffset = iCurrentOffset + iBitsightvrmapi_limit;
						}
					} catch (ex) {
						LogError("GetBitsightVRMFindings()  domain=" + sBitsightVRMDomain + " GUID=" + sTPPBitsightVRMGUID + " Axios call error. ex: " + ex);
						LogVerbose("GetBitsightVRMFindings() Axios call error. ex.stack: " + ex.stack);
					}
				} //end while loop iterating pages of Bitsight Findings API calls.
			} //end for loop of Archer TPP vendors
		} catch (ex) {
			LogError("GetBitsightVRMFindings()  domain=" + sBitsightVRMDomain + " GUID=" + sTPPBitsightVRMGUID + " Error looping this company. ex: " + ex);
			LogVerbose("GetBitsightVRMFindings() Error looping this company. ex.stack: " + ex.stack);
			//Normally we'd return false, but we will continue iterating all companies.
		}
	} catch (ex) {
		LogError("GetBitsightVRMFindings() Overall error setting up. ex: " + ex);
		LogVerbose("GetBitsightVRMFindings() Overall error setting up. ex.stack: " + ex.stack);
		return false;
	}

	//Save to file...
	if (bVerboseLogging === true) {
		var fs = require("fs");
		fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitsightVRMFindings.json", JSON.stringify(aBitsightVRMFindings));
		LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitsightVRMFindings.json file");
	}

	//Check Status based on number of findings found from Bitsight. If nothing found from Bitsight , nothing to update.
	LogInfo("GetBitsightVRMFindings() #Bitsight Findings Found: " + totalBitsightFindings);
	if (totalBitsightFindings > 0) {
		return true;
	} else {
		return false;
	}
}

async function ParseBitsightVRMVendorFindings(sBitsightVRMDomain, sTPPBitsightVRMGUID, data) {
	//Goal: Populate the aBitsightVRMFindings[] object with findings from VRM.
	//Data format documentation: https://service.Bitsighttech.com/customer-api/vrm/v1/vendors/findings/query
	//Data example: {"offset":0,"limit":10,"count":4,"results":[{"guid":"0add4431-bdc5-4459-a50c-3c3d6bd83115","review_status":"Awaiting Response","review_status_guid":"7e06eacf-4a49-4a9b-ac10-f33c39ead1c3","review_status_type":2,"vendor_guid":"96f2d60f-e86a-45ad-b5b0-ce33e1538377","created_at":"2025-02-06T16:17:47.279932","criticality":"LOW","description":"Test Finding From Bitsight Support","artifact_guid":"f1de9118-165d-4002-9e81-393bc0446ca7","will_be_resolved":null,"remediation_date":null,"last_modified":"2025-02-17 13:52:19","reported_on":"CAIQ v4 - Audit & Assurance","short_guid":"83115","linking_guids":{"survey_id":"99054c0a-a49b-4e61-8487-43cb84a3ea35","category_id":"f4f93ba8-ee22-4683-9c4a-47d4dedddf1d"},"messages_summary":null,"will_remediate_by":"pending","artifact_type":"questionnaire","is_past_due":false,"vendor":null,"enterprise_guid":"bc750dfc-1670-4ba2-9589-a813722d927d"},{"guid":"5c47aac3-f80c-40a2-a324-b3c95dfe0716","review_status":"Awaiting Response","review_status_guid":"7e06eacf-4a49-4a9b-ac10-f33c39ead1c3","review_status_type":2,"vendor_guid":"96f2d60f-e86a-45ad-b5b0-ce33e1538377","created_at":"2025-02-06T16:17:28.962468","criticality":"NONE","description":"Test Finding from Bitsight Support","artifact_guid":"472d2825-40a2-43f4-9d62-cf6333e3464d","will_be_resolved":null,"remediation_date":null,"last_modified":"2025-02-17 13:52:21","reported_on":"CAIQ v4 - Audit & Assurance","short_guid":"E0716","linking_guids":{"survey_id":"99054c0a-a49b-4e61-8487-43cb84a3ea35","category_id":"f4f93ba8-ee22-4683-9c4a-47d4dedddf1d"},"messages_summary":null,"will_remediate_by":"pending","artifact_type":"questionnaire","is_past_due":false,"vendor":null,"enterprise_guid":"bc750dfc-1670-4ba2-9589-a813722d927d"}]}

	LogInfo("ParseBitsightVRMVendorFindings() Start");

	let iNumFindingsFound = 0;

	try {
		LogInfo("ParseBitsightVRMVendorFindings() # Bitsight Findings: " + data.results.length);

		//For each result element, parse the data and add to our main object.
		//Need to make additional API calls per vendor to get the remaining data.
		for (let i in data.results) {
			try {
				//Obtain and validate data elements

				LogInfo("ParseBitsightVRMVendorFindings() Parsing domain=" + sBitsightVRMDomain + " GUID=" + sTPPBitsightVRMGUID);

				const sFindingStatus = getText(data.results[i].review_status); //Finding Status
				const sBitsightFindingID = getText(data.results[i].guid); //Finding GUID
				const sCriticality = getText(data.results[i].criticality); //Finding Criticality
				const WillRemediateBy = getText(data.results[i].will_remediate_by); //Finding Will Remediate By
				const sCreationDate = getText(data.results[i].created_at); //Finding Creation Date
				const sReportedOn = getText(data.results[i].reported_on); //Finding Reported On

				//NOTE: Using key/value pair with the vendor domain as the key.
				//Put results into the aBitsightVRMFindings[] object for this domain
				aBitsightVRMFindings[sBitsightVRMDomain].findings[aBitsightVRMFindings[sBitsightVRMDomain].findings.length] = {
					"FindingStatus": sFindingStatus,
					"Criticality": sCriticality,
					"WillRemediateBy": WillRemediateBy,
					"BitsightFindingID": sBitsightFindingID,
					"CreationDate": sCreationDate,
					"ReportedOn": sReportedOn,
				};

				iNumFindingsFound++;
			} catch (ex) {
				LogError("ParseBitsightVRMVendorFindings() Error parsing. domain=" + sBitsightVRMDomain + " ex: " + ex);
				LogVerbose("ParseBitsightVRMVendorFindings() Error parsing. ex.stack: " + ex.stack);
			}
		}
	} catch (ex) {
		LogError("ParseBitsightVRMVendorFindings() Error iterating loop. domain=" + sBitsightVRMDomain + " ex: " + ex);
		LogVerbose("ParseBitsightVRMVendorFindings() Error iterating loop. ex.stack: " + ex.stack);
	}

	//Return the number of findings found
	return iNumFindingsFound;
}

function getText(text) {
	//Simple function to validate input and return text.
	try {
		if (typeof text == "undefined" || text == null || text == "") {
			return "";
		} else {
			try {
				let s = text.toString();
				s = s.trim();
				return s;
			} catch (ex) {
				LogWarn("getText() Unable to parse text. ex:" + ex);
			}
		}
	} catch (ex) {
		LogWarn("getText() Unable to parse text. ex:" + ex);
	}
}

async function IterateDataSets() {
	//This function iterates both lists of Findings and identifies adds and updates to make in Archer.

	LogInfo("IterateDataSets() Start.");

	try {
		// Outer loop to iterate through the Bitsight Domains in the BS Findings object
		for (const domain in aBitsightVRMFindings) {
			if (typeof domain != "undefined" && domain != null && domain != "") {
				LogInfo(`IterateDataSets() Domain: ${domain}`);

				//Attempt obtaining the findings
				try {
					const aBitsightFindings = aBitsightVRMFindings[domain].findings;

					// Inner loop1 to iterate through the Bitsight findings for the current domain
					if (typeof aBitsightFindings != "undefined" && aBitsightFindings != null && aBitsightFindings.length > 0) {
						//Iterate the Findings for this domain
						for (let i = 0; i < aBitsightFindings.length; i++) {
							const aBitsightFinding = aBitsightFindings[i];
							const sBSBitsightFindingID = aBitsightFinding.BitsightFindingID;

							//Now compare to the Archer Bitsight VRM Findings
							//First goal is to find a match. If match, then compare data for updates
							//If no match, then lookup and add data to Archer
							let bFound = false;
							let iArcherFindingRecord;
							for (let iArcherFinding in aArcherBitsightVRMFindings) {
								if (aArcherBitsightVRMFindings[iArcherFinding].sBitsightFinding_BitsightFindingID == sBSBitsightFindingID) {
									bFound = true;
									iArcherFindingRecord = iArcherFinding; //need the object id in the event the finding wasn't found to add the new
									LogVerbose("IterateDataSets() Existing Finding Found. domain=" + domain + " BitsightFindingID=" + sBSBitsightFindingID);
									await CompareExistingFinding(domain, aBitsightFinding, aArcherBitsightVRMFindings[iArcherFinding]);

									break;
								}
							}

							//If the findings was not found, we need to add it.
							if (bFound === false) {
								LogVerbose("IterateDataSets() Add New finding. domain=" + domain + " BitsightFindingID=" + sBSBitsightFindingID);
								//Add Finding to object to update. Passing zero (0) as first parameter which is the Archer Content ID. Zero is new.
								await UpdateArcherFinding(0, domain, aBitsightFinding);
							}
						}
					} else {
						LogInfo("IterateDataSets() No findings for this domain.");
					}
					LogInfo("----------------------------------------------------------------------------"); // Separator for better readability
				} catch (ex) {
					LogWarn(`IterateDataSets() Findings not found. ${ex}`);
				}
			}
		}
	} catch (ex) {}
}

async function CompareExistingFinding(domain, aBitsightFinding, aArcherBitsightVRMFinding) {
	//Attempts to compare the finding from Bitsight to Archer to determine if changes are needed in Archer.

	let sBSBitsightFindingID = "TBD";

	try {
		sBSBitsightFindingID = aBitsightFinding.BitsightFindingID;

		LogInfo("CompareExistingFinding() Start. domain=" + domain + " sBSBitsightFindingID=" + sBSBitsightFindingID);

		//Get Bitsight Finding details
		const sBSFindingStatus = aBitsightFinding.FindingStatus.toLowerCase();
		const sBSCriticality = aBitsightFinding.Criticality.toLowerCase();
		const sBSWillRemediateBy = aBitsightFinding.WillRemediateBy;
		const sBSCreationDate = aBitsightFinding.CreationDate;
		const sBSReportedOn = aBitsightFinding.ReportedOn;

		//Get Archer Finding details
		const sArcherFindingStatus = aArcherBitsightVRMFinding.sBitsightFinding_FindingStatus.toLowerCase();
		const sArcherCriticality = aArcherBitsightVRMFinding.sBitsightFinding_Criticality.toLowerCase();
		const sArcherWillRemediateBy = aArcherBitsightVRMFinding.sBitsightFinding_WillRemediateBy;
		const sArcherCreationDate = aArcherBitsightVRMFinding.sBitsightFinding_CreationDate;
		const sArcherReportedOn = aArcherBitsightVRMFinding.sBitsightFinding_ReportedOn;

		let bDiffFound = false;

		//Looking for differences.
		if (sBSFindingStatus != sArcherFindingStatus) {
			LogVerbose("CompareExistingFinding() Status different! BS=" + sBSFindingStatus + " Archer=" + sArcherFindingStatus);
			bDiffFound = true;
		}
		if (sBSCriticality != sArcherCriticality) {
			LogVerbose("CompareExistingFinding() Criticality different! BS=" + sBSCriticality + " Archer=" + sArcherCriticality);
			bDiffFound = true;
		}
		if (sBSWillRemediateBy != sArcherWillRemediateBy) {
			LogVerbose("CompareExistingFinding() WillRemediateBy different! BS=" + sBSWillRemediateBy + " Archer=" + sArcherWillRemediateBy);
			bDiffFound = true;
		}
		//Need to compare apples to apples since the formatting is different between both platforms
		//BS=2025-03-28T20:56:47.121289 Archer=3/28/2025
		if (formatDateToYYYYMMDD(sBSCreationDate) != formatDateToYYYYMMDD(sArcherCreationDate)) {
			LogVerbose("CompareExistingFinding() CreationDate different! BS=" + sBSCreationDate + " Archer=" + sArcherCreationDate);
			bDiffFound = true;
		}
		if (sBSReportedOn != sArcherReportedOn) {
			LogVerbose("CompareExistingFinding() ReportedOn different! BS=" + sBSReportedOn + " Archer=" + sArcherReportedOn);
			bDiffFound = true;
		}

		if (bDiffFound === true) {
			LogInfo("CompareExistingFinding() Differences Found!");
			//await UpdateArcherFinding(aArcherBitsightVRMFinding.ArcherContentID, domain, aBitsightFinding, aArcherBitsightVRMFinding);
			await UpdateArcherFinding(aArcherBitsightVRMFinding.ArcherContentID, domain, aBitsightFinding);
		} else {
			LogInfo("CompareExistingFinding() No Differences Found.");
		}
	} catch (ex) {
		LogWarn("CompareExistingFinding() Error Checking Differences for domain=" + domain + " sBSBitsightFindingID=" + sBSBitsightFindingID + " ex: " + ex);
	}
}

function GetArcherTPPContentID(domain) {
	//This looks up the Archer Content ID based on the domain provided.

	try {
		LogInfo("UpdateArcherFinding() Start. domain=" + domain);

		if (typeof domain == "undefined" || domain == null || domain == "") {
			LogInfo("UpdateArcherFinding() Domain missing, cannot lookup. Returning false.");
			return false;
		} else {
			//Loop through the TPP info
			for (let iTPP in aArcherTPPReport) {
				//If a match is found, return the Content ID
				if (aArcherTPPReport[iTPP].BitsightVRMDomain.toLowerCase() == domain.toLowerCase()) {
					return aArcherTPPReport[iTPP].ArcherContentID;
				}
			}
		}
	} catch (ex) {
		LogWarn("UpdateArcherFinding() Error looking up domain. Returning false. ex=" + ex);
		return false;
	}
	//If we got here, a match was not found.
	LogWarn("UpdateArcherFinding() Match not found. Returning false. domain=" + domain);
	return false;
}

function lookupArcherValuesList(sFieldName, sValue) {
	LogVerbose("lookupArcherValuesList() start.  sFieldName=" + sFieldName + " sValue=" + sValue);

	try {
		if (typeof sFieldName == "undefined" || typeof sValue == "undefined" || sFieldName == null || sValue == null || sValue == "") {
			LogWarn("lookupArcherValuesList() sFieldName or sValue was empty. Returning [].");
			return [];
		} else {
			//iterate and find the list and get the info

			let archerValuesListID;
			let bFound = false;
			//Now go get the Archer valueslist ID
			for (let aVL in ArcherValuesLists) {
				if (ArcherValuesLists[aVL].FieldName.toLowerCase().trim() == sFieldName.toLowerCase().trim()) {
					for (let aVal in ArcherValuesLists[aVL].Values) {
						if (ArcherValuesLists[aVL].Values[aVal].name.toLowerCase().trim() == sValue.toLowerCase().trim()) {
							archerValuesListID = ArcherValuesLists[aVL].Values[aVal].id;
							bFound = true;
							break;
						}
					}
				}
			}

			if (bFound === true) {
				LogVerbose("lookupArcherValuesList() archerValuesListID=" + archerValuesListID);
				//Populate and return the value
				return [archerValuesListID];
			}
		}
	} catch (ex) {
		LogWarn("lookupArcherValuesList() ArcherValuesLists error parsing. Returning []. ex:" + ex);
		LogVerbose("lookupArcherValuesList() ArcherValuesLists error parsing. Returning []. ex.stack:" + ex.stack);
		return [];
	}

	//If we got here, return blank because we don't have the info we need.
	LogWarn("lookupArcherValuesList() ArcherValuesLists did not match. returning [].");
	return [];
}

//async function UpdateArcherFinding(iArcherContentID, domain, aBitsightFinding, aArcherBitsightVRMFindings) {
async function UpdateArcherFinding(iArcherContentID, domain, aBitsightFinding) {
	//Either a new finding needs to be created in Archer or an update to an existing finding is needed.
	//This function creates the correct postbody format to perform the API update in Archer.

	let bSuccess = true;

	LogInfo("UpdateArcherFinding() Start.");
	try {
		const LevelID = params["archer_VendorFindingsLevelID"];
		const ContentID = parseInt(iArcherContentID, 10);

		const sBSBitsightFindingID = aBitsightFinding.BitsightFindingID; //guid is text

		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("domain=" + domain + " sBSBitsightFindingID=" + sBSBitsightFindingID + " iArcherContentID=" + iArcherContentID);
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");
		LogInfo("************************************************************************/////////////////////////////////////////////////////////////////////////////////////");

		const iRelatedThirdPartyProfile = GetArcherTPPContentID(domain); //int

		if (iRelatedThirdPartyProfile === false) {
			LogWarn("UpdateArcherFinding() iRelatedThirdPartyProfile is empty and therefore cannot create or update the finding. Skipping.");
			return false;
		}

		const sBSWillRemediateBy = aBitsightFinding.WillRemediateBy; //text
		const sBSCreationDate = formatDateToYYYYMMDD(aBitsightFinding.CreationDate); //date needs to be in YYYY-MM-DD format
		const sBSReportedOn = aBitsightFinding.ReportedOn; //text
		const sBSFindingStatus = lookupArcherValuesList("Finding Status", aBitsightFinding.FindingStatus); //Need to format for Archer content upload into values list.
		const sBSCriticality = lookupArcherValuesList("Criticality", aBitsightFinding.Criticality);

		//Get the Archer Field IDs
		//const BSVRMFinding_ArcherID_fid = ArcherVendorFindingsFieldParams["Archer ID"].id;
		const BSVRMFinding_RelatedThirdPartyProfile_fid = ArcherVendorFindingsFieldParams["Related Third Party Profile"].id;
		const BSVRMFinding_FindingStatus_fid = ArcherVendorFindingsFieldParams["Finding Status"].id;
		const BSVRMFinding_Criticality_fid = ArcherVendorFindingsFieldParams["Criticality"].id;
		const BSVRMFinding_WillRemediateBy_fid = ArcherVendorFindingsFieldParams["Will Remediate By"].id;
		const BSVRMFinding_BitsightFindingID_fid = ArcherVendorFindingsFieldParams["Bitsight Finding ID"].id;
		const BSVRMFinding_CreationDate_fid = ArcherVendorFindingsFieldParams["Creation Date"].id;
		const BSVRMFinding_ReportedOn_fid = ArcherVendorFindingsFieldParams["Reported On"].id;

		const postBody = {
			"Content": {
				"LevelId": LevelID,
				"Tag": "Bitsight VRM Findings",
				"FieldContents": {
					[BSVRMFinding_WillRemediateBy_fid]: {
						"Type": ArcherInputTypes["text"],
						"Tag": "Will Remediate By",
						"Value": sBSWillRemediateBy,
						"FieldID": BSVRMFinding_WillRemediateBy_fid,
					},
					[BSVRMFinding_RelatedThirdPartyProfile_fid]: {
						"Type": ArcherInputTypes["crossref"],
						"Tag": "Related Third Party Profile",
						"Value": [
							{
								"ContentId": iRelatedThirdPartyProfile,
							},
						],
						"FieldID": BSVRMFinding_RelatedThirdPartyProfile_fid,
					},
					[BSVRMFinding_CreationDate_fid]: {
						"Type": ArcherInputTypes["date"],
						"Tag": "Creation Date",
						"Value": sBSCreationDate,
						"FieldID": BSVRMFinding_CreationDate_fid,
					},
					[BSVRMFinding_FindingStatus_fid]: {
						"Type": ArcherInputTypes["valueslist"],
						"Tag": "Finding Status",
						"Value": {
							"ValuesListIds": sBSFindingStatus,
						},
						"FieldID": BSVRMFinding_FindingStatus_fid,
					},
					[BSVRMFinding_Criticality_fid]: {
						"Type": ArcherInputTypes["valueslist"],
						"Tag": "Criticality",
						"Value": {
							"ValuesListIds": sBSCriticality,
						},
						"FieldID": BSVRMFinding_Criticality_fid,
					},
					[BSVRMFinding_ReportedOn_fid]: {
						"Type": ArcherInputTypes["text"],
						"Tag": "Reported On",
						"Value": sBSReportedOn,
						"FieldID": BSVRMFinding_ReportedOn_fid,
					},
					[BSVRMFinding_BitsightFindingID_fid]: {
						"Type": ArcherInputTypes["text"],
						"Tag": "Bitsight Finding ID",
						"Value": sBSBitsightFindingID,
						"FieldID": BSVRMFinding_BitsightFindingID_fid,
					},
				},
			},
		}; //End of postBody

		//Create New= POST
		//Update = PUT
		let sMethod = "";
		if (typeof ContentID == "undefined" || ContentID == 0 || ContentID == null || ContentID == "") {
			sMethod = "POST";
		} else {
			//Add the Archer record ID to the postbody
			postBody.Content.Id = ContentID;
			sMethod = "PUT";
		}

		LogVerbose("UpdateArcherFinding() postbody=" + JSON.stringify(postBody));

		try {
			let data = null;

			const sUrl = params["archer_webroot"] + params["archer_rest_root"] + params["archer_contentpath"];

			const httpConfig = {
				method: sMethod,
				url: sUrl,
				headers: {
					"Authorization": 'Archer session-id="' + sArcherSessionToken + '"',
					"Content-Type": "application/json",
					Accept: "application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
				},
				data: postBody,
			};

			LogVerbose("UpdateArcherFinding() API call httpConfig=" + JSON.stringify(httpConfig));

			await axios(httpConfig)
				.then(function (res) {
					data = res.data;
					LogVerbose("UpdateArcherFinding() Axios call complete. data=" + JSON.stringify(data));
				})
				.catch(function (error) {
					bSuccess = false;
					totalErrorUpdate++;
					LogError("UpdateArcherFinding() Axios call error. Err: " + error);
					LogVerbose("UpdateArcherFinding() Axios call error. Err.stack: " + error.stack);
				});

			if (
				typeof data != "undefined" &&
				typeof data.RequestedObject != "undefined" &&
				typeof data.RequestedObject.Id != "undefined" &&
				typeof data.IsSuccessful != "undefined" &&
				data.IsSuccessful == true
			) {
				LogVerbose("UpdateArcherFinding() Archer Finding update successful. ID=" + data.RequestedObject.Id);
				totalSuccessUpdate++;
			} else {
				bSuccess = false;
				totalErrorUpdate++;
				LogError("UpdateArcherFinding() ERROR updating Finding. API result not sucessful.");
			}
		} catch (ex) {
			bSuccess = false;
			totalErrorUpdate++;
			LogError("UpdateArcherFinding() Error constructing call or parsing result. ex: " + ex);
			LogVerbose("UpdateArcherFinding() Error constructing call or parsing result. ex.stack: " + ex.stack);
		}
	} catch (ex) {
		bSuccess = false;
		totalErrorUpdate++;
		LogError("UpdateArcherFinding() Error creating Archer postbody for " + domain + " ex:" + ex);
		LogVerbose("UpdateArcherFinding() Error creating Archer postbody for " + domain + " ex.stack:" + ex.stack);
	}
	return bSuccess;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////END CORE FUNCTIONALITY
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Main driver function of the overall process
async function main() {
	//flag of successful transactions just for inside this loop.
	let bSuccess = true;
	LogInfo("main() Launching...");

	//Login to Archer
	bSuccess = await ArcherLogin();

	//If we successfully received reports continue
	if (bSuccess === true) {
		LogInfo("main() - Got Archer Session token.");
		//Get Archer Version for Bitsight Stats - Not critical and if it fails, will not fail the entire process
		await GetArcherVersion();

		//Get the Archer App ID and Fields - Needed to construct search criteria and update the Archer record after Adding to Bitsight.
		bSuccess = await getArcherTPPFieldIDs();

		if (bSuccess === true) {
			bSuccess = await getVendorFindingsFieldIDs(); //This also gets all the values list values

			if (bSuccess === true) {
				LogInfo("getVendorFindingsFieldIDs successful, make requests to Bitsight to get VRM Finding Status values.");

				bSuccess = await getBitsightListValues();

				if (bSuccess === true) {
					LogInfo("getBitsightListValues successful, ensure they match Archer or update Archer.");
					bSuccess = await ValidateAndUpdateListValues();

					if (bSuccess === true) {
						bSuccess = await getArcherTPPsWithDomain();

						if (bSuccess === true) {
							bSuccess = await getArcherVRMFindings();

							if (bSuccess === true) {
								bSuccess = await GetBitsightVRMFindings();

								if (bSuccess === true) {
									bSuccess = await IterateDataSets();

									// 	if (bSuccess === true) {
									// 		bSuccess = await UpdateArcherVRMFindings();
									// 	}
								}
							}
						}
					}
				}
			}
		}
	} else if (bSuccess === false) {
		LogInfo("main() - Error obtaining Archer Session Token. Failing overall.");
	}
}

//This function runs the overall process.
async function runAwait() {
	try {
		//Launch main process/functionality
		await main();
	} catch (ex) {
		LogInfo("==================================================================================");
		LogError("runAwait() Error Executing main(). Err: " + ex);
		LogError("runAwait() Error Executing main(). Err.stack: " + ex.stack);
		LogInfo("==================================================================================");
	} finally {
		//Wrap up and show final Archer Original Requested object which has the status:
		LogInfo("==================================================================================");
		//TODO: LogInfo("runAwait() Final aArcherTPPReport: " + JSON.stringify(aArcherTPPReport));
		LogInfo("==================================================================================");

		//Wrap up and show final object Bitsight object which has the status:
		LogInfo("==================================================================================");
		//TODO: LogInfo("runAwait() Final BitsightVRMVendors: " + JSON.stringify(BitsightVRMVendors));
		LogInfo("==================================================================================");

		//Show final stats
		LogInfo("==================================STATS===========================================");
		LogInfo("            totalArcherTPPs: " + totalArcherTPPs.toString());
		LogInfo("totalArcherBitsightFindings: " + totalArcherBitsightFindings.toString());
		LogInfo("      totalBitsightFindings: " + totalBitsightFindings.toString());
		LogInfo("         totalSuccessUpdate: " + totalSuccessUpdate.toString());
		LogInfo("           totalErrorUpdate: " + totalErrorUpdate.toString());
		LogInfo("                totalErrors: " + totalErrors.toString());
		LogInfo("              totalWarnings: " + totalWarnings.toString());

		LogInfo("==================================================================================");
		//Sanity check for matches. We should have the same numbers for found and added.
		// if (bOverallSuccessful === true && totalReportRequests == totalReportSuccess) {
		// 	LogInfo("runAwait() No Errors found and MATCH! Total found equals updated!");
		// 	//We could consider this a success if they match, but could be a false sense of accomplishment.
		// 	//Since we set bOverallSuccessful if a LogError() is called, honor that.
		// 	bOverallSuccessful = true;
		// } else {
		// 	LogInfo("runAwait() ERROR or MISMATCH! Total found does NOT equal updated!");
		// 	bOverallSuccessful = false;
		// }

		LogInfo("==================================================================================");
		LogInfo("==================================================================================");
		LogInfo("runAwait() Summary of Errors/Warnings from debugLog:\r\n" + sErrorLogDetails);
		LogInfo("==================================================================================");
		LogInfo("==================================================================================");

		if (bOverallSuccessful == true) {
			LogSaaSSuccess();
			LogInfo("runAwait() Finished SUCCESSFULLY");
		} else {
			try {
				LogSaaSError();
			} catch (ex) {
				LogInfo("runAwait() ERROR LogSaaSError. SaaSErr02: " + ex);
				LogInfo("runAwait() ERROR LogSaaSError. SaaSErr02: stack=" + ex.stack);
			}
			LogInfo("runAwait() Finished with ERRORS");
		}
	}
}

//start async app driver for overall functionality and then determines final success/fail resolution.
runAwait();
